/* Copyright (c) 2012,2014, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/module.h>
#include <linux/slab.h>
#include <linux/err.h>
#include <linux/wcnss_wlan.h>
#include <linux/spinlock.h>

static DEFINE_SPINLOCK(alloc_lock);

struct wcnss_prealloc {
	int occupied;
	unsigned int size;
	void *ptr;
};

/* pre-alloced mem for WLAN driver */
static struct wcnss_prealloc wcnss_allocs[] = {
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 8  * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 12 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 16 * 1024, NULL},
	{0, 24 * 1024, NULL},
	{0, 24 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 32 * 1024, NULL},
	{0, 64 * 1024, NULL},
	{0, 64 * 1024, NULL},
	{0, 76 * 1024, NULL},
};

int wcnss_prealloc_init(void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(wcnss_allocs); i++) {
		wcnss_allocs[i].occupied = 0;
		wcnss_allocs[i].ptr = kmalloc(wcnss_allocs[i].size, GFP_KERNEL);
		if (wcnss_allocs[i].ptr == NULL)
			return -ENOMEM;
	}

	return 0;
}

void wcnss_prealloc_deinit(void)
{
	int i = 0;

	for (i = 0; i < ARRAY_SIZE(wcnss_allocs); i++) {
		kfree(wcnss_allocs[i].ptr);
		wcnss_allocs[i].ptr = NULL;
	}
}

void *wcnss_prealloc_get(unsigned int size)
{
	int i = 0;
	unsigned long flags;

	spin_lock_irqsave(&alloc_lock, flags);
	for (i = 0; i < ARRAY_SIZE(wcnss_allocs); i++) {
		if (wcnss_allocs[i].occupied)
			continue;

		if (wcnss_allocs[i].size > size) {
			/* we found the slot */
			wcnss_allocs[i].occupied = 1;
			spin_unlock_irqrestore(&alloc_lock, flags);
			return wcnss_allocs[i].ptr;
		}
	}
	spin_unlock_irqrestore(&alloc_lock, flags);
	pr_err("wcnss: %s: prealloc not available for size: %d\n",
			__func__, size);

	return NULL;
}
EXPORT_SYMBOL(wcnss_prealloc_get);

int wcnss_prealloc_put(void *ptr)
{
	int i = 0;
	unsigned long flags;

	spin_lock_irqsave(&alloc_lock, flags);
	for (i = 0; i < ARRAY_SIZE(wcnss_allocs); i++) {
		if (wcnss_allocs[i].ptr == ptr) {
			wcnss_allocs[i].occupied = 0;
			spin_unlock_irqrestore(&alloc_lock, flags);
			return 1;
		}
	}
	spin_unlock_irqrestore(&alloc_lock, flags);

	return 0;
}
EXPORT_SYMBOL(wcnss_prealloc_put);

static int __init wcnss_pre_alloc_init(void)
{
	return wcnss_prealloc_init();
}

static void __exit wcnss_pre_alloc_exit(void)
{
	wcnss_prealloc_deinit();
}

module_init(wcnss_pre_alloc_init);
module_exit(wcnss_pre_alloc_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION(DEVICE "WCNSS Prealloc Driver");
